//******************************************************************************
//   MSP430x26x Demo - SMBus Slave Writes to MSP430 Master with PEC
//
//   Description: This demo connects two MSP430's via the I2C bus.  The master 
//   reads from the slave. This is the slave code. The master code is called
//   msp430x26x_SMB_mstr_PEC.c. The TX data begins at 1 and is incremented 
//   each time it is sent. PEC is appended with each byte
//   The USCI_B0 TX interrupt is used to know when to TX.
//   ACLK = n/a, MCLK = SMCLK = BRCLK = DCO ~ 1.1MHz
//   //* USCI Required *//
//   
//                                 /|\  /|\
//                  MSP430F2619     10k  10k     MSP430F2619
//                    slave         |    |        master           
//              -----------------|  |    |  ----------------- 
//             |             P3.1|<-|----+>|P3.1             |
//             |                 |  |      |             P1.0|-->LED
//             |                 |  |      |                 |
//             |             P3.2|<-+----->|P3.2             |
//             |                 |         |                 |
//
//
//  H. Grewal
//  Texas Instruments Inc.
//  June 2009
//  Built with IAR Embedded Workbench Version: 4.11B
//******************************************************************************

#include  <msp430x26x.h>
#define CRC_MSG_SIZE	2                   // # of Bytes to calculate CRC
#define CRC8_POLY	    0x07                // X^8 + X^2 + X + 1
#define CRC8_INIT_REM   0x0                 // Initial Remainder

typedef enum {
             SMBS_MISC               = (unsigned char) 0x48,
             SMBS_DEVICE_DFLT        = (unsigned char) 0x61,
             SMBS_ALERT_RESPONSE     = (unsigned char) 0x0C,
             SMBS_HOST               = (unsigned char) 0x08,
             SMBS_10BIT_ADD          = (unsigned char) 0x78,
             SMBS_DFLT_ADD           = (unsigned char) 0x37,
             SMBS_ACCESS_HOST        = (unsigned char) 0x28,             
             SMBS_CBUS               = (unsigned char) 0x01,
             SMBS_GCAL               = (unsigned char) 0x00
             }SMBS_Address_t;               
/*---------------------------------------------------------------------------*/	 

void SMBS_Init (SMBS_Address_t Add_Param,unsigned char Slave_Add);

unsigned char crc8MakeBitwise(unsigned char crc, unsigned short poly, unsigned char *pmsg, unsigned int msg_size);
volatile unsigned char TXByteCtr;           // # of bytes TXed
unsigned char crc8;                         // CRC-8 result
unsigned char msg[CRC_MSG_SIZE] = {0};      // Array to store Address,Data


unsigned char Transmit = 0;                 // Data to be TXed

void main (void)
{
  WDTCTL = WDTPW + WDTHOLD;                 // Stop WDT
  P3SEL |= 0x06;                            // Select I2C pins
  P1DIR |= 0xFF;                            // P1.0 output
                                            // unused ports to output
  P1OUT =  0;
  P2DIR =  0xFF;
  P3DIR =  0xFF;
  P4DIR =  0xFF;
  P5DIR =  0xFF;
  P6DIR =  0xFF; 
  SMBS_Init (SMBS_MISC,0x48);               // Own address configured as 48h
  while (1)
  {
    Transmit++;
    TXByteCtr = 0;                          // Clear TX byte count    
    __bis_SR_register(LPM4_bits+GIE);       // Enter LPM4 w/ interrupts
  }
}

void SMBS_Init (SMBS_Address_t Add_Param,unsigned char Slave_Add)
{
  UCB0CTL1 |= UCSWRST;                      // Enable SW reset
  UCB0CTL0 = UCMODE_3 + UCSYNC;             // I2C Slave, synchronous mode
  if (Add_Param == SMBS_MISC)
  {
      UCB0I2COA = Slave_Add;                // Own Address is 048h
  } 
  else
      UCB0I2COA = (unsigned char) Add_Param ;
  UCB0CTL1 &= ~UCSWRST;                     // Clear SW reset, resume operation  

  UCB0I2CIE |= UCSTPIE + UCSTTIE;           // Enable STT and STP interrupt
  IE2 |= UCB0TXIE;                          // Enable TX interrupt
}

// CRC-8 Algorithm to check the PEC
unsigned char crc8MakeBitwise(unsigned char crc, unsigned short poly, unsigned char *pmsg, unsigned int msg_size)
{
   volatile unsigned char i, j, carry = 0;

    for(i = 0 ; i < msg_size ; i ++)
    {
       crc ^= *pmsg++;

       for(j = 0 ; j < 8 ; j++)
       {
  
         carry = crc & 0x80;
         if (carry)
         {
          crc <<= 1;
          crc ^= poly;
         }

         else
         crc <<= 1;
       }
    }
   
    return(crc);
}       

//------------------------------------------------------------------------------
// The USCI_B0 data ISR is used to move data from MSP430 memory to the
// I2C master. PTxData points to the next byte to be transmitted, and TXByteCtr
// keeps track of the number of bytes transmitted.
//------------------------------------------------------------------------------
#pragma vector = USCIAB0TX_VECTOR
__interrupt void USCIAB0TX_ISR(void)
{
 
   
   msg[0] = UCB0I2COA;                      // CRC calculated on Address,R/W
   msg[1] = Transmit;                       // and Data
   if (TXByteCtr)
   {
    crc8 = crc8MakeBitwise(CRC8_INIT_REM, CRC8_POLY, msg, CRC_MSG_SIZE);
    UCB0TXBUF = crc8;                       // Transmit CRC-8
   }
   else
   {
    UCB0TXBUF = Transmit;                   // Transmit Data
   }
   TXByteCtr++;
 
}

//------------------------------------------------------------------------------
// The USCI_B0 state ISR is used to wake up the CPU from LPM0 in order to do
// processing in the main program after data has been transmitted. LPM0 is
// only exit in case of a (re-)start or stop condition when actual data
// was transmitted.
//------------------------------------------------------------------------------
#pragma vector = USCIAB0RX_VECTOR
__interrupt void USCIAB0RX_ISR(void)
{
  UCB0STAT &= ~(UCSTPIFG + UCSTTIFG);       // Clear interrupt flags
  if (TXByteCtr)                            // Check TX byte counter
  {
    __bic_SR_register_on_exit(LPM4_bits);   // LPM4 exit after STP
  }
}                                           

